package com.logicbig.example;

import org.junit.jupiter.api.*;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.*;
import java.util.List;
import java.util.stream.Stream;
import static org.junit.jupiter.api.Assertions.*;

public class NestedWithParameterizedTestsTest {
    private String systemName;
    
    @BeforeEach
    void setUpSystem() {
        systemName = "TestSystem";
    }
    
    @Test
    void systemInitializationTest() {
        assertEquals("TestSystem", systemName);
    }
    
    @Nested
    class MathOperations {
        
        @ParameterizedTest
        @ValueSource(ints = {1, 2, 3, 4, 5})
        void squareTest(int number) {
            int result = number * number;
            assertTrue(result > 0);
            assertEquals(Math.pow(number, 2), result, 0.001);
        }
        
        @ParameterizedTest
        @CsvSource({"1, 2, 3", "5, 10, 15", "-3, 3, 0"})
        void additionTest(int a, int b, int expected) {
            assertEquals(expected, a + b);
        }
        
        @Nested
        class AdvancedMath {
            
            @ParameterizedTest
            @MethodSource("powerDataProvider")
            void powerTest(int base, int exponent, double expected) {
                assertEquals(expected, Math.pow(base, exponent), 0.001);
            }
            
            static Stream<Arguments> powerDataProvider() {
                return Stream.of(
                    Arguments.of(2, 3, 8.0),
                    Arguments.of(5, 2, 25.0),
                    Arguments.of(10, 0, 1.0)
                );
            }
            
            @ParameterizedTest
            @CsvFileSource(resources = "/test-data.csv", numLinesToSkip = 1)
            void csvFileTest(int input, int expected) {
                assertEquals(expected, input * 2);
            }
        }
    }
    
    @Nested
    class StringOperations {
        private String prefix = "TEST_";
        
        @BeforeEach
        void setUpStringContext() {
            prefix = "TEST_";
        }
        
        @ParameterizedTest
        @ValueSource(strings = {"hello", "world", "junit", "testing"})
        void stringLengthTest(String input) {
            assertTrue(input.length() > 0);
            assertEquals(input.length(), input.length());
        }
        
        @ParameterizedTest
        @CsvSource({"hello, HELLO", "jUnit, JUNIT", "test, TEST"})
        void uppercaseTest(String input, String expected) {
            assertEquals(expected, input.toUpperCase());
        }
        
        @ParameterizedTest
        @MethodSource("prefixedStringProvider")
        void prefixedStringTest(String input, String expected) {
            String result = prefix + input;
            assertEquals(expected, result);
        }
        
        static Stream<Arguments> prefixedStringProvider() {
            return Stream.of(
                Arguments.of("data", "TEST_data"),
                Arguments.of("value", "TEST_value"),
                Arguments.of("input", "TEST_input")
            );
        }
        
        @Nested
        class StringValidation {
            
            @ParameterizedTest
            @NullAndEmptySource
            @ValueSource(strings = {"  ", "\t", "\n"})
            void blankStringTest(String input) {
                assertTrue(input == null || input.trim().isEmpty());
            }
            
            @ParameterizedTest
            @EnumSource(value = TestEnum.class, names = {"ACTIVE", "PENDING"})
            void enumTest(TestEnum status) {
                assertNotNull(status);
                assertTrue(status == TestEnum.ACTIVE || status == TestEnum.PENDING);
            }
        }
    }
    
    @Nested
    class DataCombinationTests {
        
        @ParameterizedTest
        @CsvSource({"2, true", "3, false", "4, true", "5, false"})
        void evenOddTest(int number, boolean isEven) {
            if (isEven) {
                assertEquals(0, number % 2);
            } else {
                assertEquals(1, number % 2);
            }
        }
        
        @ParameterizedTest
        @MethodSource("rangeDataProvider")
        void rangeTest(int start, int end, int expectedCount) {
            long count = java.util.stream.IntStream.range(start, end).count();
            assertEquals(expectedCount, count);
        }
        
        static Stream<Arguments> rangeDataProvider() {
            return Stream.of(
                Arguments.of(1, 5, 4),
                Arguments.of(0, 10, 10),
                Arguments.of(-5, 5, 10)
            );
        }
    }
}

enum TestEnum {
    ACTIVE, INACTIVE, PENDING, COMPLETED
}