package com.logicbig.example;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.model.chat.ChatModel;
import dev.langchain4j.model.chat.request.ChatRequest;
import dev.langchain4j.model.chat.request.ResponseFormat;
import dev.langchain4j.model.chat.request.ResponseFormatType;
import dev.langchain4j.model.chat.request.json.*;
import dev.langchain4j.model.chat.response.ChatResponse;
import dev.langchain4j.model.ollama.OllamaChatModel;
import java.util.Arrays;
import java.util.List;

public class StatusEnumExample {
    public static void main(String[] args) {
        // Create Ollama ChatModel with phi3
        ChatModel chatModel =
                OllamaChatModel.builder()
                               .baseUrl("http://localhost:11434")
                               .modelName("phi3:mini-128k")
                               .logRequests(true)
                               .logResponses(true)
                               .build();


        // Create enum schema for status
        JsonSchemaElement statusSchema =
                JsonEnumSchema.builder()
                              .enumValues(Arrays.stream(Status.values())
                                                .map(Enum::name)
                                                .toList())
                              .description("Current status of the item")
                              .build();

        // Create object schema with status field
        JsonSchemaElement itemSchema = JsonObjectSchema.builder()
                                                       .addStringProperty("itemName")
                                                       .addProperty("status", statusSchema)
                                                       .required("itemName", "status")
                                                       .build();

        JsonSchemaElement itemsArraySchema = JsonArraySchema.builder()
                                                            .items(itemSchema)
                                                            .description("List of items with their statuses")
                                                            .build();
        // Create JSON Schema
        JsonSchema jsonSchema = JsonSchema.builder()
                                          .name("ItemStatusList")
                                          .rootElement(itemsArraySchema)
                                          .build();


        // Create response format
        ResponseFormat responseFormat = ResponseFormat.builder()
                                                      .type(ResponseFormatType.JSON)
                                                      .jsonSchema(jsonSchema)
                                                      .build();

        // Text describing status
        String text = """
                The 'Website Redesign' is currently active and progressing well.
                The 'Database Migration' has been completed successfully.
                The 'User Authentication' is still pending review.
                """;

        UserMessage userMessage = UserMessage.from(text);

        // Create chat request
        ChatRequest chatRequest = ChatRequest.builder()
                                             .responseFormat(responseFormat)
                                             .messages(userMessage)
                                             .build();


        System.out.println("Extracting status information...");

        // Send request and get response
        ChatResponse chatResponse = chatModel.chat(chatRequest);

        // Extract JSON response
        String jsonResponse = chatResponse.aiMessage().text();
        System.out.println("JSON Response: " + jsonResponse);

        ObjectMapper mapper = new ObjectMapper();
        try {
            List<ItemStatus> items =
                    mapper.readValue(jsonResponse,
                                     new TypeReference<List<ItemStatus>>() {});

            for (ItemStatus item : items) {
                System.out.println(item);
            }

        } catch (Exception e) {
            System.err.println("Error parsing JSON: " + e.getMessage());
        }
    }
}