package com.logicbig.example;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.model.chat.ChatModel;
import dev.langchain4j.model.chat.request.ChatRequest;
import dev.langchain4j.model.chat.request.ResponseFormat;
import dev.langchain4j.model.chat.request.ResponseFormatType;
import dev.langchain4j.model.chat.request.json.JsonArraySchema;
import dev.langchain4j.model.chat.request.json.JsonObjectSchema;
import dev.langchain4j.model.chat.request.json.JsonSchema;
import dev.langchain4j.model.chat.request.json.JsonSchemaElement;
import dev.langchain4j.model.chat.response.ChatResponse;
import dev.langchain4j.model.ollama.OllamaChatModel;
import java.util.List;

public class PeopleArrayExample {
    public static void main(String[] args) {
        // Create Ollama ChatModel with phi3
        ChatModel chatModel =
                OllamaChatModel.builder()
                               .baseUrl("http://localhost:11434")
                               .modelName("phi3:mini-128k")
                               .logRequests(true)
                               .logResponses(true)
                               .build();

        // Define schema for a single person
        JsonSchemaElement personSchema = JsonObjectSchema.builder()
                                                         .addStringProperty("name")
                                                         .addIntegerProperty("age")
                                                         .required("name", "age")
                                                         .build();

        // Create array schema for multiple people
        JsonSchemaElement peopleArraySchema = JsonArraySchema.builder()
                                                             .items(personSchema)
                                                             .build();

        // Create JSON Schema with array as root
        JsonSchema jsonSchema = JsonSchema.builder()
                                          .name("People")
                                          .rootElement(peopleArraySchema)
                                          .build();

        // Create response format
        ResponseFormat responseFormat = ResponseFormat.builder()
                                                      .type(ResponseFormatType.JSON)
                                                      .jsonSchema(jsonSchema)
                                                      .build();

        // Text containing multiple people
        String text = """
                In the meeting we had John who is 42 years old, 
                Sarah who is 35 years old, and Michael who is 28 years old.
                Also present was Lisa, aged 31.
                """;

        UserMessage userMessage = UserMessage.from(text);

        // Create chat request
        ChatRequest chatRequest = ChatRequest.builder()
                                             .responseFormat(responseFormat)
                                             .messages(userMessage)
                                             .build();


        // Send request and get response
        ChatResponse chatResponse = chatModel.chat(chatRequest);

        // Extract JSON response
        String jsonResponse = chatResponse.aiMessage().text();
        System.out.println("JSON Response: " + jsonResponse);

        // Parse JSON to List of Person objects
        ObjectMapper mapper = new ObjectMapper();
        try {
            List<Person> people = mapper.readValue(jsonResponse,
                                                   new TypeReference<List<Person>>() {});

            System.out.println("\nExtracted People:");
            for (Person person : people) {
                System.out.println(person);
            }
        } catch (Exception e) {
            System.err.println("Error parsing JSON: " + e.getMessage());
        }
    }
}