package com.logicbig.example;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import dev.langchain4j.data.message.AiMessage;
import dev.langchain4j.data.message.ChatMessage;
import dev.langchain4j.data.message.SystemMessage;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.store.memory.chat.ChatMemoryStore;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

public class FileChatMemoryStore implements ChatMemoryStore {
    private final Path file;
    private final ObjectMapper mapper = new ObjectMapper();

    FileChatMemoryStore(Path file) {
        this.file = file;
    }

    @Override
    public List<ChatMessage> getMessages(Object memoryId) {
        try {
            if (!Files.exists(file) || Files.size(file) == 0) {
                return new ArrayList<>();
            }

            List<JsonEntry> entries =
                    mapper.readValue(file.toFile(),
                                     new TypeReference<>() {});

            List<ChatMessage> messages = new ArrayList<>();

            for (JsonEntry entry : entries) {
                switch (entry.type) {
                    case "userMessage" ->
                            messages.add(UserMessage.from(entry.message));
                    case "systemMessage" ->
                            messages.add(SystemMessage.from(entry.message));
                    case "aiMessage" ->
                            messages.add(AiMessage.from(entry.message));
                }
            }

            return messages;

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void updateMessages(Object memoryId, List<ChatMessage> newMessages) {
        try {
            List<JsonEntry> entries = new ArrayList<>();

            // append new entries
            for (ChatMessage message : newMessages) {
                entries.add(toEntry(message));
            }

            mapper.writerWithDefaultPrettyPrinter()
                  .writeValue(file.toFile(), entries);

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void deleteMessages(Object memoryId) {
        try {
            Files.deleteIfExists(file);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private JsonEntry toEntry(ChatMessage message) {
        if (message instanceof UserMessage) {
            return new JsonEntry("userMessage", ((UserMessage) message).singleText());
        }
        if (message instanceof SystemMessage) {
            return new JsonEntry("systemMessage", ((SystemMessage) message).text());
        }
        if (message instanceof AiMessage) {
            return new JsonEntry("aiMessage", ((AiMessage) message).text());
        }
        throw new IllegalArgumentException("Unsupported message type");
    }

    static class JsonEntry {
        public String type;
        public String message;

        // required by Jackson
        public JsonEntry() {
        }

        JsonEntry(String type, String message) {
            this.type = type;
            this.message = message;
        }
    }
}