package com.logicbig.example;

import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.extension.AfterEachCallback;
import org.junit.jupiter.api.extension.BeforeAllCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;
import java.util.Optional;

public class DependencyInjectionExtension implements BeforeEachCallback,
        AfterEachCallback, BeforeAllCallback {

    @Override
    public void beforeAll(ExtensionContext context) {
        System.out.println("\n=== BEFORE ALL ===");

        // Check the test instance lifecycle
        Optional<TestInstance.Lifecycle> lifecycle = context.getTestInstanceLifecycle();
        lifecycle.ifPresent(life -> {
            System.out.println("Test Instance Lifecycle: " + life);
            if (life == TestInstance.Lifecycle.PER_CLASS) {
                System.out.println("[INFO] Single instance shared across all test methods");
            } else {
                System.out.println("[INFO] New instance for each test method");
            }
        });
    }

    @Override
    public void beforeEach(ExtensionContext context) {
        System.out.println("\n=== BEFORE EACH ===");

        // Get the test instance (will be present for test methods)
        context.getTestInstance().ifPresent(instance -> {
            System.out.println("Test Instance: " + instance.getClass().getSimpleName());
            System.out.println("Instance Hash: " + System.identityHashCode(instance));

            // Inject a service dependency into the test instance
            if (instance instanceof InjectableTest test) {
                // Create and inject a mock service based on test requirements
                String serviceName = "Service-" + context.getDisplayName();
                MockService service = new MockService(serviceName);
                test.setService(service);

                System.out.println("[INJECTED] Service: " + service.getName());
                System.out.println("[INJECTED] Service ID: " + service.getId());

                // Store service reference for cleanup
                context.getStore(ExtensionContext.Namespace.create(instance))
                       .put("injected-service", service);
            }
        });

        context.getTestInstances().ifPresent(instances -> {
            System.out.println("Test Instances:");
            instances.getAllInstances().forEach(instance ->
                                                        System.out.println(instance.getClass().getSimpleName())
            );
            System.out.println("");
        });
    }

    @Override
    public void afterEach(ExtensionContext context) {
        System.out.println("\n=== AFTER EACH ===");

        // Clean up resources associated with the test instance
        context.getTestInstance().ifPresent(instance -> {
            // Retrieve and clean up the injected service
            Object service = context.getStore(ExtensionContext.Namespace.create(instance))
                                    .remove("injected-service");

            if (service instanceof MockService) {
                MockService mockService = (MockService) service;
                System.out.println("[CLEANUP] Service cleaned up: " + mockService.getName());
                mockService.cleanup();
            }

            System.out.println("Test Instance Hash (cleanup): " + System.identityHashCode(instance));
        });

        // Demonstrate required instance access (fails fast if no instance)
        try {
            Object requiredInstance = context.getRequiredTestInstance();
            System.out.println("[VERIFIED] Required instance available");
        } catch (IllegalStateException e) {
            System.out.println("[WARNING] No test instance available: " + e.getMessage());
        }
    }

}