package com.logicbig.example;

import org.junit.jupiter.api.extension.*;
import org.junit.jupiter.api.parallel.ExecutionMode;
import java.lang.reflect.Method;
import java.util.Set;

public class MetadataAwareExtension implements BeforeEachCallback {
    
    @Override
    public void beforeEach(ExtensionContext context) {
        System.out.println("\n=== TEST METADATA ANALYSIS ===");
        
        // 1. Analyze the element (Class/Method) and its annotations
        analyzeElement(context);
        
        // 2. Check and process tags
        processTags(context);
        
        // 3. Check execution mode for thread safety
        checkExecutionMode(context);
        
        // 4. Apply configuration based on metadata
        applyConfiguration(context);
    }
    
    private void analyzeElement(ExtensionContext context) {
        System.out.println("Element Analysis:");
        
        // Get the annotated element (method or class)
        context.getElement().ifPresent(element -> {
            System.out.println("  Element Type: " + element.getClass().getSimpleName());
            System.out.println("  Element Name: " + element.toString());
            
            // Check for custom annotations
            if (element.isAnnotationPresent(AuditLog.class)) {
                AuditLog annotation = element.getAnnotation(AuditLog.class);
                System.out.println("  Found @AuditLog");
                System.out.println("    Level: " + annotation.level());
                System.out.println("    Category: " + annotation.category());
                System.out.println("    Track Arguments: " + annotation.trackArguments());
            }
        });
        
        // Get test class and method specifically
        context.getTestClass().ifPresent(testClass -> {
            System.out.println("  Test Class: " + testClass.getSimpleName());
        });

        //instead of getTestClass() You can use:
        try {
            Class<?> testClass = context.getRequiredTestClass();
            System.out.println("  Required Test Class: " + testClass.getSimpleName());
        } catch (IllegalStateException e) {
            System.out.println("  No test class available: " + e.getMessage());
        }
        
        context.getTestMethod().ifPresent(method -> {
            System.out.println("  Test Method: " + method.getName());
            System.out.println("  Return Type: " + method.getReturnType().getSimpleName());
            System.out.println("  Parameter Count: " + method.getParameterCount());
        });

        //instead of getTestMethod() You can use:
        try {
            Method method = context.getRequiredTestMethod();
            System.out.println("  Required Test Method: " + method.getName());
            System.out.println("  Required TestMethod Return Type: " +
                                       method.getReturnType().getSimpleName());
        } catch (IllegalStateException e) {
            System.out.println("  No test method available: " + e.getMessage());
        }
    }

    private void processTags(ExtensionContext context) {
        Set<String> tags = context.getTags();
        System.out.println("Tag Analysis:");
        
        if (tags.isEmpty()) {
            System.out.println("  No tags found");
            return;
        }
        
        System.out.println("  Tags: " + String.join(", ", tags));
        
        // Apply conditional logic based on tags
        if (tags.contains("slow")) {
            System.out.println("  ⚠ SLOW TEST: Enabling extended monitoring");
        }
        
        if (tags.contains("integration")) {
            System.out.println("  INTEGRATION TEST: Database setup required");
        }
        
        if (tags.contains("security")) {
            System.out.println("  SECURITY TEST: Enabling audit logging");
        }
    }
    
    private void checkExecutionMode(ExtensionContext context) {
        ExecutionMode mode = context.getExecutionMode();
        System.out.println("Execution Mode:");
        System.out.println("  Mode: " + mode);
        
        switch (mode) {
            case SAME_THREAD:
                System.out.println("  Single-threaded execution");
                break;
            case CONCURRENT:
                System.out.println("  Concurrent execution - thread safety required");
                System.out.println("  Using thread-local storage for test data");
                break;
        }
    }
    
    private void applyConfiguration(ExtensionContext context) {
        System.out.println("Applied Configuration:");
        
        // Enable audit logging if annotation is present
        boolean auditEnabled = false;
        String auditLevel = "INFO";
        String auditCategory = "general";
        
        context.getElement().ifPresent(element -> {
            if (element.isAnnotationPresent(AuditLog.class)) {
                AuditLog annotation = element.getAnnotation(AuditLog.class);
                System.out.println("    Audit logging enabled");
                System.out.println("    Log Level: " + annotation.level());
                System.out.println("    Category: " + annotation.category());
                
                if (annotation.trackArguments()) {
                    System.out.println("    Argument tracking: Enabled");
                }
            }
        });
        
        // Security enhancements for security-tagged tests
        Set<String> tags = context.getTags();
        if (tags.contains("security")) {
            System.out.println("    Security enhancements enabled");
            System.out.println("    Input validation: Strict mode");
            System.out.println("    Log sanitization: Enabled");
        }
        
        // Thread safety measures for concurrent execution
        if (context.getExecutionMode() == ExecutionMode.CONCURRENT) {
            System.out.println("    Concurrent execution safeguards");
            System.out.println("    Resource locking: Enabled");
            System.out.println("    Test isolation: Per-thread");
        }
        
        System.out.println("---");
    }
}