package com.logicbig.example;

import org.junit.jupiter.api.extension.AfterTestExecutionCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;
import java.util.Optional;

public class FailureAnalysisExtension implements AfterTestExecutionCallback, BeforeEachCallback {

    private static final String ATTEMPT_KEY = "retry-attempt";

    @Override
    public void beforeEach(ExtensionContext context) {
        // Initialize retry attempt counter
        context.getStore(ExtensionContext.Namespace
                                 .create(context.getRequiredTestMethod()))
               .put(ATTEMPT_KEY, 1);
    }

    @Override
    public void afterTestExecution(ExtensionContext context) {
        Optional<Throwable> executionException = context.getExecutionException();

        System.out.println("\n=== TEST EXECUTION ANALYSIS ===");
        System.out.println("Test: " + context.getDisplayName());

        if (executionException.isPresent()) {
            Throwable exception = executionException.get();
            System.out.println("[FAILURE] Test failed with exception: " +
                                       exception.getClass().getSimpleName());
            System.out.println("[FAILURE] Message: " + exception.getMessage());

            // Enhanced failure reporting
            generateFailureReport(context, exception);

            // Check if retry should be attempted
            if (shouldRetryTest(context, exception)) {
                attemptRetry(context, exception);
            }
        } else {
            System.out.println("[SUCCESS] Test passed successfully");

            // Log success details for diagnostics
            logSuccessDetails(context);
        }

        System.out.println("---");
    }

    private void generateFailureReport(ExtensionContext context,
                                       Throwable exception) {
        System.out.println("[REPORT] Enhanced Failure Analysis:");
        System.out.println("  Exception Type: " + exception.getClass().getName());
        System.out.println("  Exception Message: " + exception.getMessage());

        // Show first line of stack trace for context
        StackTraceElement[] stackTrace = exception.getStackTrace();
        if (stackTrace.length > 0) {
            System.out.println("  Location: " + stackTrace[0]);
        }

        // Check for specific exception types
        if (exception instanceof NetworkTimeoutException) {
            System.out.println("  [DIAGNOSTIC] Network timeout detected");
            System.out.println("  [DIAGNOSTIC] Consider increasing timeout or checking network");
        } else if (exception instanceof NullPointerException) {
            System.out.println("  [DIAGNOSTIC] Null pointer exception");
            System.out.println("  [DIAGNOSTIC] Check for uninitialized objects");
        } else if (exception instanceof IllegalArgumentException) {
            System.out.println("  [DIAGNOSTIC] Invalid argument provided");
            System.out.println("  [DIAGNOSTIC] Validate input parameters");
        }
    }

    private boolean shouldRetryTest(ExtensionContext context,
                                    Throwable exception) {
        // Check if test method has @RetryOnException annotation
        return context.getTestMethod().map(method -> {
            if (method.isAnnotationPresent(RetryOnException.class)) {
                RetryOnException annotation = method.getAnnotation(RetryOnException.class);

                // Check if exception type matches retry configuration
                for (Class<? extends Throwable> retryException : annotation.value()) {
                    if (retryException.isInstance(exception)) {
                        System.out.println("[RETRY] Exception " + exception.getClass().getSimpleName() +
                                                   " matches retry configuration");
                        return true;
                    }
                }
            }
            return false;
        }).orElse(false);
    }

    private void attemptRetry(ExtensionContext context,
                              Throwable exception) {
        context.getTestMethod().ifPresent(method -> {
            RetryOnException annotation = method.getAnnotation(RetryOnException.class);

            // Get current attempt count
            ExtensionContext.Store store = context.getStore(
                    ExtensionContext.Namespace.create(method)
            );
            int currentAttempt = store.get(ATTEMPT_KEY, Integer.class);

            if (currentAttempt < annotation.maxAttempts()) {
                System.out.println("[RETRY] Attempt " + currentAttempt +
                                           " failed, retrying (max " + annotation.maxAttempts() + ")");

                // Increment attempt counter
                store.put(ATTEMPT_KEY, currentAttempt + 1);

                // Simulate delay before retry
                try {
                    Thread.sleep(annotation.delayMillis());
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }

                // Throw special exception to trigger retry
                // In real implementation, you would use JUnit's retry mechanism
                // or a custom test template
                System.out.println("[RETRY] Would retry test here (implementation detail)");
            } else {
                System.out.println("[RETRY] Max attempts (" + annotation.maxAttempts() + ") reached");
                System.out.println("[RETRY] Test will be marked as failed");
            }
        });
    }

    private void logSuccessDetails(ExtensionContext context) {
        System.out.println("[DIAGNOSTIC] Test execution details:");
        context.getTestClass().ifPresent(clazz -> {
            System.out.println("  Test Class: " + clazz.getSimpleName());
        });
        context.getTestMethod().ifPresent(method -> {
            System.out.println("  Test Method: " + method.getName());
        });
    }
}