package com.logicbig.example;

import org.junit.jupiter.api.extension.*;
import java.lang.reflect.Constructor;
import java.util.UUID;

public class OrderTestInstanceFactory implements TestInstanceFactory {
    
    @Override
    public Object createTestInstance(
            TestInstanceFactoryContext factoryContext,
            ExtensionContext extensionContext) {
        
        Class<?> testClass = factoryContext.getTestClass();
        System.out.println("[FACTORY] Creating test instance for: " + testClass.getSimpleName());
        
        // Find the constructor with @OrderId parameter
        Constructor<?> constructor = null;
        for (Constructor<?> c : testClass.getDeclaredConstructors()) {
            if (c.getParameterCount() == 1 && 
                c.getParameters()[0].isAnnotationPresent(OrderId.class)) {
                constructor = c;
                break;
            }
        }
        
        if (constructor == null) {
            throw new IllegalStateException(
                "No constructor with @OrderId parameter found in " + testClass.getSimpleName());
        }
        
        System.out.println("[FACTORY] Using constructor: " + constructor);
        
        // Generate dynamic tracking information
        String instanceId = UUID.randomUUID().toString().substring(0, 8);
        System.out.println("[FACTORY] Generated instance ID: " + instanceId);
        
        // Store tracking info in ExtensionContext for later use
        extensionContext.getStore(ExtensionContext.Namespace.GLOBAL)
                        .put("instanceId", instanceId);
        
        // Use ExecutableInvoker to create the instance
        // Parameters will be resolved by registered ParameterResolvers
        System.out.println("[FACTORY] Invoking constructor via ExecutableInvoker");
        Object instance = extensionContext.getExecutableInvoker()
                .invoke(constructor, null);  // null because top-level class
        
        System.out.println("[FACTORY] Instance created successfully");
        System.out.println("[FACTORY] Instance hash: " + System.identityHashCode(instance));
        
        return instance;
    }
}