package com.logicbig.example;

import org.junit.jupiter.api.extension.AfterTestExecutionCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.ExtensionContext.Namespace;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

public class ConfigurableRetryExtension implements BeforeEachCallback, AfterTestExecutionCallback {

    private static final String MAX_ATTEMPTS_KEY = "retry.maxAttempts";
    private static final String DELAY_KEY = "retry.delay";
    private static final String RETRY_ON_KEY = "retry.retryOn";

    private static final String ATTEMPT_KEY = "retry-attempt";
    private static final Namespace NAMESPACE =
            Namespace.create(ConfigurableRetryExtension.class);

    @Override
    public void beforeEach(ExtensionContext context) {
        // Initialize attempt counter
        context.getStore(NAMESPACE).put(ATTEMPT_KEY, 1);

        // Read and log configuration
        readConfiguration(context);
    }

    @Override
    public void afterTestExecution(ExtensionContext context) {
        Optional<Throwable> executionException = context.getExecutionException();

        if (executionException.isPresent()) {
            Throwable exception = executionException.get();

            // Read configuration values
            int maxAttempts = readMaxAttempts(context);
            long delay = readDelay(context);
            List<String> retryOnExceptions = readRetryOnExceptions(context);

            // Check if exception should trigger retry
            if (shouldRetry(exception, retryOnExceptions)) {
                int currentAttempt = context.getStore(NAMESPACE).get(ATTEMPT_KEY, Integer.class);

                if (currentAttempt < maxAttempts) {
                    System.out.println("[RETRY] Attempt " + currentAttempt +
                                               " failed with " + exception.getClass().getSimpleName());
                    System.out.println("[RETRY] Retrying (max " + maxAttempts +
                                               " attempts, delay " + delay + "ms)");

                    // Increment attempt counter
                    context.getStore(NAMESPACE).put(ATTEMPT_KEY, currentAttempt + 1);

                    // Simulate retry delay
                    try {
                        Thread.sleep(delay);
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }

                    // In real implementation, you would re-execute the test
                    System.out.println("[RETRY] Would retry test here");
                } else {
                    System.out.println("[RETRY] Max attempts (" + maxAttempts + ") reached");
                }
            } else {
                System.out.println("[RETRY] Exception " + exception.getClass().getSimpleName() +
                                           " not in retry list");
            }
        }
    }

    private void readConfiguration(ExtensionContext context) {
        System.out.println("[CONFIG] Reading configuration parameters:");

        // Read with transformation
        Optional<Integer> maxAttempts = context.getConfigurationParameter(MAX_ATTEMPTS_KEY, Integer::valueOf);
        maxAttempts.ifPresent(value ->
                                      System.out.println("  " + MAX_ATTEMPTS_KEY + " = " + value)
        );

        Optional<Long> delay = context.getConfigurationParameter(DELAY_KEY, Long::valueOf);
        delay.ifPresent(value ->
                                System.out.println("  " + DELAY_KEY + " = " + value + "ms")
        );

        Optional<List<String>> retryOn =
                context.getConfigurationParameter(RETRY_ON_KEY,
                                                  value -> Arrays.asList(value.split(",")));
        retryOn.ifPresent(list ->
                                  System.out.println("  " + RETRY_ON_KEY + " = " + list)
        );

        // Show missing configuration
        if (maxAttempts.isEmpty()) {
            System.out.println("  " + MAX_ATTEMPTS_KEY + " = (not set, using default: 2)");
        }
        if (delay.isEmpty()) {
            System.out.println("  " + DELAY_KEY + " = (not set, using default: 100ms)");
        }
        if (retryOn.isEmpty()) {
            System.out.println("  " + RETRY_ON_KEY + " = (not set, using default: [])");
        }
    }

    private int readMaxAttempts(ExtensionContext context) {
        return context.getConfigurationParameter(MAX_ATTEMPTS_KEY, Integer::valueOf)
                      .orElse(2); // Default value
    }

    private long readDelay(ExtensionContext context) {
        return context.getConfigurationParameter(DELAY_KEY, Long::valueOf)
                      .orElseThrow(() -> new RuntimeException(
                              "'%s' not provided".formatted(DELAY_KEY)));
    }

    private List<String> readRetryOnExceptions(ExtensionContext context) {
        return context.getConfigurationParameter(RETRY_ON_KEY,
                                                 value -> Arrays.asList(value.split(",")))
                      .orElse(Collections.emptyList());
    }

    private boolean shouldRetry(Throwable exception,
                                List<String> retryOnExceptions) {
        String exceptionName = exception.getClass().getSimpleName();
        return retryOnExceptions.stream()
                                .anyMatch(exceptionName::contains);
    }
}