package com.logicbig.example.engine;


import org.junit.platform.engine.*;
import org.junit.platform.engine.discovery.ClassSelector;
import org.junit.platform.engine.discovery.PackageSelector;
import org.junit.platform.engine.support.descriptor.EngineDescriptor;
import org.junit.platform.suite.api.SelectPackages;
import org.junit.platform.suite.api.Suite;

import java.lang.reflect.Modifier;
import java.util.Set;

public class MyCustomTestEngine implements TestEngine {

    public static final String ENGINE_ID = "my-simple-test-engine";

    @Override
    public String getId() {
        return ENGINE_ID;
    }

    @Override
    public TestDescriptor discover(
            EngineDiscoveryRequest discoveryRequest,
            UniqueId uniqueId
    ) {

        EngineDescriptor engineDescriptor =
                new EngineDescriptor(uniqueId, "My Custom Test Engine");

        discoveryRequest
                .getSelectorsByType(ClassSelector.class)
                .forEach(selector -> handleSuiteClass(selector, engineDescriptor));

        return engineDescriptor;
    }



    private void handleSuiteClass(
            ClassSelector selector,
            EngineDescriptor engineDescriptor
    ) {

        Class<?> suiteClass = selector.getJavaClass();

        SelectPackages selectPackages =
                suiteClass.getAnnotation(SelectPackages.class);

        if (selectPackages == null) {
            return;
        }

        for (String pkg : selectPackages.value()) {
            discoverPackage(pkg, engineDescriptor);
        }
    }

    private void discoverPackage(
            String packageName,
            EngineDescriptor engineDescriptor
    ) {

        if (!packageName.startsWith("com.logicbig.example.tests")) {
            return;
        }

        Set<Class<?>> candidates =
                ClasspathScanner.findImplementations(
                        packageName,
                        CustomTest.class
                );

        for (Class<?> testClass : candidates) {
            try {
                Object instance =
                        testClass.getDeclaredConstructor().newInstance();

                UniqueId id = engineDescriptor.getUniqueId()
                                              .append("custom-test", testClass.getName());

                engineDescriptor.addChild(
                        new CustomTestDescriptor(
                                id,
                                testClass.getSimpleName(),
                                instance
                        )
                );

            } catch (Exception ignored) {
            }
        }
    }


    @Override
    public void execute(ExecutionRequest request) {

        TestDescriptor root = request.getRootTestDescriptor();
        EngineExecutionListener listener =
                request.getEngineExecutionListener();

        listener.executionStarted(root);

        for (TestDescriptor descriptor : root.getChildren()) {

            CustomTestDescriptor testDescriptor =
                    (CustomTestDescriptor) descriptor;

            listener.executionStarted(testDescriptor);

            try {
                CustomTest test =
                        (CustomTest) testDescriptor.getTestInstance();

                boolean passed = test.performTest();

                if (!passed) {
                    throw new AssertionError(
                            "CustomTest.performTest() returned false"
                    );
                }

                listener.executionFinished(
                        testDescriptor,
                        TestExecutionResult.successful()
                );

            } catch (Throwable ex) {
                listener.executionFinished(
                        testDescriptor,
                        TestExecutionResult.failed(ex)
                );
            }
        }

        listener.executionFinished(
                root,
                TestExecutionResult.successful()
        );
    }
}